/* symtab.c
   Time-stamp: "97/01/03 17:48:58 mik"

   Copyright (C) 1991, 92, 93, 94, 95, 96, 97
	Christian Schenk  <cschenk@berlin.snafu.de>

   This file is part of MiKTeX.

   MiKTeX is free software; you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.
   
   MiKTeX is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.
   
   You should have received a copy of the GNU General Public License
   along with MiKTeX; if not, write to the Free Software Foundation,
   59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <limits.h>

#include "common.h"
#include "gram.h"

static void new_keyword (const char *, unsigned);
static void new_type (const char *, unsigned, void *);
static void new_variable (const char *, const char *);
static void new_constant (const char *, const char *, long);
static void new_function (const char *, const char *);
static void new_procedure (const char *);
static void new_build_in (const char *, unsigned);

extern long yylineno;
extern unsigned block_level;

static char char_pool[CHAR_POOL_SIZE];

static STRING_PTR string_ptr;
static STRING_PTR string_mark;

STRING_PTR
new_string (const char *s)

{
  int l = strlen (s) + 1;
  STRING_PTR ret = string_ptr;

  if (string_ptr + l > CHAR_POOL_SIZE)
    c4p_error ("internal error: must increase `CHAR_POOL_SIZE'");
  strcpy (&char_pool[string_ptr], s);
  string_ptr += l;
  return (ret);
}

const char *
get_string (STRING_PTR s)

{
  if (s >= string_ptr)
    c4p_error ("internal error: get_string: string pointer out of range");
  return (&char_pool[s]);
}

static SYMBOL_PTR hash_table[HASH_PRIME];

static SYMBOL_PTR
hash (const char *id)

{
  const char *p = id;
  unsigned long h = 0;
  unsigned long g;

  for (; *p; p++)
    {
      h = (h << 4) + *p;
      g = h & 0xf000000;
      if (g)
	{
	  h ^= (g >> 24);
	  h |= 0x0ffffff;
	}
    }

  return ((SYMBOL_PTR) (h % HASH_PRIME));
}

#define NIL_SYMBOL (MAX_SYMBOLS + 1)

static symbol_t symbol_table[MAX_SYMBOLS];

static SYMBOL_PTR symbol_ptr;
static SYMBOL_PTR symbol_mark;

symbol_t *
new_symbol_instance (const char *id)

{
  SYMBOL_PTR h = hash (id);
  SYMBOL_PTR front = hash_table[h];
  symbol_t *new_symbol;

  if (symbol_ptr == MAX_SYMBOLS)
    c4p_error ("internal error: must increase `MAX_SYMBOLS'");

  new_symbol = &symbol_table[symbol_ptr];
  hash_table[h] = symbol_ptr;
  new_symbol->s_kind = UNDEFINED_IDENTIFIER;
  new_symbol->s_next = front;
  new_symbol->s_type = UNKNOWN_TYPE;
  new_symbol->s_key = symbol_ptr;
  new_symbol->s_repr = id;
  new_symbol->s_flags = 0;
  new_symbol->s_translated_type = 0;
  symbol_ptr++;

  return (new_symbol);
}

static symbol_t *
insert (const char *id,
	SYMBOL_PTR h)

{
  SYMBOL_PTR front = hash_table[h];
  symbol_t *new_symbol;

  if (symbol_ptr == MAX_SYMBOLS)
    c4p_error ("internal error: must increase `MAX_SYMBOLS'");

  new_symbol = &symbol_table[symbol_ptr];
  hash_table[h] = symbol_ptr;
  new_symbol->s_kind = UNDEFINED_IDENTIFIER;
  new_symbol->s_block_level = block_level;
  new_symbol->s_next = front;
  new_symbol->s_type = UNKNOWN_TYPE;
  new_symbol->s_key = symbol_ptr;
  new_symbol->s_repr = &char_pool[string_ptr];
  new_symbol->s_flags = 0;
  new_string (id);
  symbol_ptr++;

  return (new_symbol);
}

symbol_t *
lookup (const char *id)

{
  SYMBOL_PTR h = hash (id);
  SYMBOL_PTR s = hash_table[h];
  symbol_t *possible_result = 0;

  for (; s < NIL_SYMBOL; s = symbol_table[s].s_next)
    if (strcmp (symbol_table[s].s_repr, id) == 0) /* FIXME: ignore case */
      {
	possible_result = &symbol_table[s];
	if (possible_result->s_kind == C_RESERVED)
	  return (lookup (possible_result->s_type_ptr));
	if (possible_result->s_kind != FIELD_IDENTIFIER)
	  return (possible_result);
      }
  if (possible_result)
    return (possible_result);
  return (insert (id, h));
}

void
mark_symbol_table (void)

{
  symbol_mark = symbol_ptr;
}

void
mark_string_table (void)

{
  string_mark = string_ptr;
}

void
unmark_symbol_table (void)

{
  SYMBOL_PTR h = HASH_PRIME;
  SYMBOL_PTR s;

  while (h--)
    {
      s = hash_table[h];
      while (s >= symbol_mark && s != NIL_SYMBOL)
	{
	  if (symbol_table[s].s_kind == CONSTANT_IDENTIFIER)
	    out_form ("#undef %s\n", symbol_table[s].s_repr);
	  s = symbol_table[s].s_next;
	}
      hash_table[h] = s;
    }

  symbol_ptr = symbol_mark;
}

void
unmark_string_table (void)

{
  string_ptr = string_mark;
}

void
symtab_init (void)

{
  array_node *arr;
  unsigned i;

  for (i = 0; i < HASH_PRIME; i++)
    hash_table[i] = NIL_SYMBOL;

  /* Pascal-Schluesselwoerter */
  new_keyword ("and", AND);
  new_keyword ("array", ARRAY);
  new_keyword ("begin", BEGIn);
  new_keyword ("case", CASE);
  new_keyword ("const", CONST);
  new_keyword ("div", DIV);
  new_keyword ("do", DO);
  new_keyword ("downto", DOWNTO);
  new_keyword ("else", ELSE);
  new_keyword ("end", END);
  new_keyword ("file", FILe);
  new_keyword ("for", FOR);
  new_keyword ("forward", FORWARD);
  new_keyword ("function", FUNCTION);
  new_keyword ("goto", GOTO);
  new_keyword ("if", IF);
  new_keyword ("in", IN);
  new_keyword ("label", LABEL);
  new_keyword ("mod", MOD);
  new_keyword ("nil", NIL);
  new_keyword ("not", NOT);
  new_keyword ("of", OF);
  new_keyword ("or", OR);
  new_keyword ("others", OTHERS);
  new_keyword ("packed", PACKED);
  new_keyword ("procedure", PROCEDURE);
  new_keyword ("program", PROGRAM);
  new_keyword ("record", RECORD);
  new_keyword ("repeat", REPEAT);
  new_keyword ("set", SET);
  new_keyword ("then", THEN);
  new_keyword ("to", TO);
  new_keyword ("type", TYPE);
  new_keyword ("until", UNTIL);
  new_keyword ("var", VAR);
  new_keyword ("while", WHILE);
  new_keyword ("with", WITH);

  /* Hide C keywords */
  new_mapping ("auto", "_c4p_auto");
  new_mapping ("break", "_c4p_break");
  new_mapping ("continue", "_c4p_continue");
  new_mapping ("default", "_c4p_default");
  new_mapping ("double", "_c4p_double");
  new_mapping ("enum", "_c4p_enum");
  new_mapping ("float", "_c4p_float");
  new_mapping ("free", "_c4p_free");
  new_mapping ("int", "_c4p_int");
  new_mapping ("long", "_c4p_long");
  new_mapping ("main", "_c4p_main");
  new_mapping ("register", "_c4p_register");
  new_mapping ("return", "_c4p_return");
  new_mapping ("short", "_c4p_short");
  new_mapping ("signed", "_c4p_signed");
  new_mapping ("sizeof", "_c4p_sizeof");
  new_mapping ("static", "_c4p_static");
  new_mapping ("struct", "_c4p_struct");
  new_mapping ("switch", "_c4p_switch");
  new_mapping ("typedef", "_c4p_typedef");
  new_mapping ("union", "_c4p_union");
  new_mapping ("unsigned", "_c4p_unsigned");
  new_mapping ("void", "_c4p_void");
  new_mapping ("volatile", "_c4p_volatile");

  new_type ("integer", INTEGER_TYPE, 0);
  new_type ("char", CHARACTER_TYPE, 0);
  new_type ("boolean", BOOLEAN_TYPE, 0);
  new_type ("real", REAL_TYPE, 0);
  new_type ("longreal", LONG_REAL_TYPE, 0);
  new_type ("text", FILE_NODE,
	    new_type_node (FILE_NODE, CHARACTER_TYPE, 0));

  new_type ("c4pstring", STRING_TYPE, 0);
  arr = new_type_node (ARRAY_NODE, 0L, 0L);
  arr->component_type = STRING_TYPE;
  new_type ("c4pstrarr", ARRAY_NODE, arr);
  new_type ("c4pjmpbuf", STRING_TYPE, 0);

  new_variable ("input", "text");
  new_variable ("output", "text");

  new_variable ("c4pargc", "integer");
  new_variable ("c4pargv", "c4pstrarr");
  new_variable ("c4pplen", "integer");
  new_variable ("c4ppline", "c4pstring");

  new_variable ("c4phour", "integer");
  new_variable ("c4pminute", "integer");
  new_variable ("c4pday", "integer");
  new_variable ("c4pmonth", "integer");
  new_variable ("c4pyear", "integer");

  new_variable ("c4pcur", "integer"); /* --> SEEK_CUR */
  new_variable ("c4pend", "integer"); /* --> SEEK_END */
  new_variable ("c4pset", "integer"); /* --> SEEK_SET */
  new_variable ("c4prmode", "integer");	/* --> "r" */
  new_variable ("c4pwmode", "integer");	/* --> "w" */
  new_variable ("c4prbmode", "integer"); /* --> "rb" */
  new_variable ("c4pwbmode", "integer"); /* --> "wb" */

  new_function ("abs", "integer");
  new_function ("chr", "char");
  new_function ("eof", "boolean");
  new_function ("eoln", "boolean");
  new_function ("odd", "boolean");
  new_function ("ord", "integer");
  new_function ("round", "integer");
  new_function ("trunc", "integer");

  new_procedure ("get");
  new_procedure ("put");
  new_procedure ("reset");
  new_procedure ("rewrite");

  new_function ("c4pferror", "integer"); /* ferror () */
  new_function ("c4pfopen", "boolean");	/* fopen () */
  new_function ("c4ptryfopen", "boolean"); /* fopen () */
  new_function ("c4pftell", "integer");	/* ftell () */
  new_function ("c4pinteger", "integer"); /* (integer) */
  new_function ("c4pptr", "integer"); /* & */
  new_function ("c4psetjmp", "integer"); /* setjmp () */
  new_function ("c4pstrlen", "integer");

  new_procedure ("c4pbufwrite"); /* fwrite () */
  new_procedure ("_c4p_break");	/* fflush () */
  new_procedure ("c4pexit");	/* exit () */
  new_procedure ("c4pfclose");	/* fclose () */
  new_procedure ("c4pfopen");	/* fopen () */
  new_procedure ("c4pfseek");	/* fseek () */
  new_procedure ("c4parrcpy");	/* memcpy () */
  new_procedure ("c4pmemcpy");	/* memcpy () */
  new_procedure ("c4pstrcpy");	/* strcpy () */
  new_procedure ("c4plongjmp");	/* longjmp () */
  new_procedure ("c4pgetc");	/* getc () */
  new_procedure ("c4pputc");	/* putc () */
  new_procedure ("c4pincr");	/* ++ */
  new_procedure ("c4pdecr");	/* -- */
  new_procedure ("c4pmget");

  new_build_in ("read", READ);
  new_build_in ("readln", READLN);
  new_build_in ("write", WRITE);
  new_build_in ("writeln", WRITELN);

  new_constant ("false", "boolean", 0);
  new_constant ("true", "boolean", 1);
  new_constant ("maxint", "integer", LONG_MAX);	/* FIXME: not portable */
}

static void
new_keyword (const char *keyword,
	     unsigned number)

{
  symbol_t *sym = new_symbol_instance (keyword);
  sym->s_block_level = 0;
  sym->s_kind = PASCAL_KEYWORD;
  sym->s_type = number;
}

static void
new_type (const char *type_name,
	  unsigned type,
	  void *type_ptr)

{
  symbol_t *sym = new_symbol_instance (type_name);
  sym->s_block_level = 0;
  sym->s_kind = TYPE_IDENTIFIER;
  sym->s_type = type;
  sym->s_type_ptr = type_ptr;
}

static void
new_variable (const char *var_name,
	      const char *type)
{
  symbol_t *sym = new_symbol_instance (var_name);
  symbol_t *tsym = lookup (type);

  sym->s_block_level = 0;
  sym->s_kind = VARIABLE_IDENTIFIER;
  sym->s_type = tsym->s_type;
  sym->s_type_ptr = tsym->s_type_ptr;
}

static void
new_function (const char *func_name,
	      const char *result_type)

{
  symbol_t *sym = new_symbol_instance (func_name);
  sym->s_block_level = 0;
  sym->s_kind = FUNCTION_IDENTIFIER;
  sym->s_type = PROTOTYPE_NODE;
  sym->s_type_ptr = new_type_node (PROTOTYPE_NODE, sym, 0,
				   lookup (result_type));
}

static void
new_build_in (const char *name,
	      unsigned number)

{
  symbol_t *sym = new_symbol_instance (name);
  sym->s_block_level = 0;
  sym->s_kind = BUILD_IN_IDENTIFIER;
  sym->s_type = number;
}

static void
new_procedure (const char *proc_name)

{
  symbol_t *sym = new_symbol_instance (proc_name);
  sym->s_block_level = 0;
  sym->s_kind = PROCEDURE_IDENTIFIER;
  sym->s_type = PROTOTYPE_NODE;
  sym->s_type_ptr = new_type_node (PROTOTYPE_NODE, sym, 0, 0);
}

void
new_mapping (const char *name,
	     const char *mapped_name)

{
  symbol_t *sym = lookup (name);
  sym->s_kind = C_RESERVED;
  sym->s_type_ptr = (void *) get_string (new_string (mapped_name));
}

static void
new_constant (const char *constant_name,
	      const char *type,
	      long ivalue)

{
  symbol_t *sym = new_symbol_instance (constant_name);
  symbol_t *tsym = lookup (type);
  sym->s_block_level = 0;
  sym->s_kind = CONSTANT_IDENTIFIER;
  sym->s_type = tsym->s_type;
  sym->s_value.ivalue = ivalue;
}

static unsigned long counter;

symbol_t *
new_pseudo_symbol (void)

{
  char pseudo_name[33];
  sprintf (pseudo_name, "_c4p_P%lu", counter++);
  return (lookup (pseudo_name));
}

symbol_t *
define_symbol (symbol_t *sym,
	       unsigned kind,
	       unsigned block_level,
	       unsigned type,
	       void *type_ptr,
	       value_t *value)

{
  if (sym->s_kind != UNDEFINED_IDENTIFIER)
    {
      if (block_level == sym->s_block_level &&
	  sym->s_kind != FIELD_IDENTIFIER)
	c4p_error ("`%s' already defined", sym->s_repr);
      if (sym->s_kind == CONSTANT_IDENTIFIER)
	{
	  /* FIXME: should #undefine symbol
	           out_form ("#undef %s\n", sym -> s_repr);
	  */
	}
      sym = new_symbol_instance (sym->s_repr);
    }

  sym->s_kind = kind;
  sym->s_block_level = block_level;
  sym->s_type = type;
  sym->s_type_ptr = type_ptr;
  if (value)
    sym->s_value = *value;

  return (sym);
}

int n_fast_vars;
static const char *fast_vars[MAX_FAST_VARS];

void
new_fast_var (const char *name)

{
  if (n_fast_vars == MAX_FAST_VARS)
    c4p_error ("internal error: too many fast variables");
  fast_vars[n_fast_vars++] = name;
}

int
is_fast_var (const char *name)

{
  int i;

  for (i = 0; i < n_fast_vars; i++)
    if (strcmp (fast_vars[i], name) == 0)
      return (1);
  return (0);
}
