/* MakeTeXPK.c -- Make a PK file
   Time-stamp: "97/08/24 11:21:16 mik"

   Copyright (C) 1995, 96, 97
	Christian Schenk  <cschenk@berlin.snafu.de>

   This file is part of MiKTeX.

   MiKTeX is free software; you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.
   
   MiKTeX is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.
   
   You should have received a copy of the GNU General Public License
   along with MiKTeX; if not, write to the Free Software Foundation,
   59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  */

#include <stdlib.h>
#include <stdio.h>
#include <getopt.h>
#include <io.h>
#include <direct.h>
#include <process.h>

#include "miktex.h"
#include "MakeTeXPK.rc"

#if defined (_WIN32)
#  define WIN32_LEAN_AND_MEAN
#  include <windows.h>
#endif /* _WIN32 */

#define PATH_SEP_CHAR ';'
#define isslash(c) ((c) == '\\' || (c) == '/')

static int		verbose_p;
static int		debug_p;
static int		quiet_p;
static int		print_only_p;

static const char *	destdir;

static char *		name;
static int		dpi;
static int		bdpi;
static char *		magnification;
static char *		mode;

static void
EXIT (int code)

{
  exit (code);
}

/* _________________________________________________________________________

   USAGE / HELP
   _________________________________________________________________________ */

static void
show_version (void)

{
  printf ("This is MakeTeXPK version %s (%s %s)\n",
	  VER_FILEVERSION_STR, VER_PRODUCTNAME_STR,
	  VER_PRODUCTVERSION_STR);
}

static void
usage (void)

{
  puts ("Usage: MakeTeXPK [OPTION]... name dpi bdpi magnification [MODE]");
  puts ("");
  puts ("This program makes a PK font.");
  puts ("");
  puts ("NAME is the name of the font, such as `cmr10'.  DPI is");
  puts ("the resolution the font is needed at.  BDPI is the base");
  puts ("resolution, useful for figuring out the mode to make the font");
  puts ("in.  MAGNIFICATION is a string to pass to MF as the");
  puts ("magnification.  MODE, if supplied, is the mode to use.");
  puts ("");
  puts ("Options:");
  puts ("--debug, -d                         Print debugging information.");
  puts ("--dest-dir DIRECTORY, -D DIRECTORY  Specify the destination font directory.");
  puts ("--help, -h                          Print this help screen and exit.");
  puts ("--print-only, -n                    Print what commands would be executed.");
  puts ("--verbose, -v                       Print information on what is being done.");
  puts ("--version, -V                       Print the version number and exit.");
  puts ("");
  puts ("Email problem reports to cschenk@berlin.snafu.de");
}

/* _________________________________________________________________________

   GETTING COMMAND LINE OPTIONS.
   _________________________________________________________________________ */

static struct option	long_options[] =

{
  "debug",		no_argument,		0,	'd',
  "help",		no_argument,		0,	'h',
  "print-only",		no_argument,		0,	'n',
  "dest-dir",		required_argument,	0,	'D',
  "quiet",		no_argument,		0,	'q',
  "verbose",		no_argument,		0,	'v',
  "version",		no_argument,		0,	'V',
  0,			no_argument,		0,	0,
};

static void
get_options (int	argc,
	     char **	argv)

{
  int option_index;
  int c;

  destdir = 0;
  print_only_p = 0;
  quiet_p = 0;
#if defined (DEBUG)
  verbose_p = 1;
  debug_p = 1;
#else
  verbose_p = 0;
  debug_p = 0;
#endif

  while ((c=getopt_long (argc, argv, "dD:hnqvV", long_options, &option_index))
	 != EOF)
    {
      switch (c)
	{
	case 'h':
	  usage ();
	  EXIT (0);
	  break;
	case 'n':
	  print_only_p = 1;
	  break;
	case 'D':
	  destdir = strdup (optarg);
	  break;
	case 'd':
	  debug_p = 1;
	  break;
	case 'v':
	  verbose_p = 1;
	  break;
	case 'q':
	  quiet_p = 1;
	  break;
	case 'V':
	  show_version ();
	  EXIT (0);
	  break;
	default:
	  usage ();
	  EXIT (1);
	  break;
	}
    }

  if (argc - optind < 4)
    {
      usage ();
      EXIT (1);
    }

  name = strdup (argv[optind++]);
  dpi = atoi (argv[optind++]);
  bdpi = atoi (argv[optind++]);
  magnification = strdup (argv[optind++]);

  mode = 0;
  if (optind < argc)
    mode = strdup (argv[optind++]);
  
  if (optind != argc)
    {
      usage ();
      EXIT (1);
    }
}

/* _________________________________________________________________________

   CREATING THE DESTINATION DIRECTORY.
   _________________________________________________________________________ */


static char *
create_dest_directory (const char *	name,
		       int		bdpi,
		       int		dpi,
		       const char *	mode)

{
  char dest_directories[_MAX_PATH];
  char name_template[_MAX_FNAME];
  char supplier[_MAX_FNAME];
  char typeface[_MAX_FNAME];
  char *cp_dest_directories;
  const char *cp_name_template;
  char *destdir;
  char *next_destdir;
  char *result;

  if (! get_font_info (name, supplier, typeface))
    {
      strcpy (supplier, "public");
      strcpy (typeface, "xxx");
    }

  get_cfg_value ("MakeTeXPK", "PK Temp Dir",
		 name_template, sizeof (name_template),
		 "%R\\fonts\\pk\\%m\\%s\\%t\\dpi%d");

  cp_name_template = name_template;
  cp_dest_directories = dest_directories;
  
  while (*cp_name_template)
    {
      if (*cp_name_template == '%')
	{
	  switch (cp_name_template[1])
	    {
	    default:
	      *cp_dest_directories = 0;
	      break;
	    case '%':
	      *cp_dest_directories++ = '%';
	      *cp_dest_directories = 0;
	      break;
	    case 'R':
	      *cp_dest_directories++ = '%';
	      *cp_dest_directories++ = 'R';
	      *cp_dest_directories = 0;
	      break;
	    case 'b':
	      sprintf (cp_dest_directories, "%d", bdpi);
	      break;
	    case 'd':
	      sprintf (cp_dest_directories, "%d", dpi);
	      break;
	    case 'm':
	      strcpy (cp_dest_directories, mode);
	      break;
	    case 's':
	      strcpy (cp_dest_directories, supplier);
	      break;
	    case 't':
	      strcpy (cp_dest_directories, typeface);
	      break;
	    }
	  cp_dest_directories += strlen (cp_dest_directories);
	  cp_name_template += 2;
	}
      else
	*cp_dest_directories++ = *cp_name_template++;
    }
  *cp_dest_directories = 0;

  /* Initialize search. */
  destdir = dest_directories;
  next_destdir = strchr (destdir, PATH_SEP_CHAR);
  if (next_destdir)
    *next_destdir++ = 0; 

  result = 0;
  while (destdir && result == 0)
    {
      if (destdir[0] == '%' && destdir[1] == 'R' && isslash (destdir[2]))
	{
	  char alt_destdir[_MAX_PATH];
	  size_t i;

	  for (i = 0; i < get_number_of_texmf_roots () && result == 0; i++)
	    {
	      xassert (get_root_directory (i) != 0);
	      _makepath (alt_destdir, 0, get_root_directory (i),
			 &destdir[3], 0);
	      if (print_only_p)
		{
		  printf ("mkdir %s\n", alt_destdir);
		  result = strdup (alt_destdir);
		}
	      else if (make_path_ex (alt_destdir, 0))
		result = strdup (alt_destdir);
	    }
	}
      else if (print_only_p)
	{
	  printf ("mkdir %s\n", destdir);
	  result = strdup (destdir);
	}
      else if (make_path_ex (destdir, 0))
	result = strdup (destdir);
      
      if (next_destdir && *next_destdir)
	{
	  destdir = next_destdir;
	  next_destdir = strchr (next_destdir, PATH_SEP_CHAR);
	  if (next_destdir)
	    *next_destdir++ = 0;
	}
      else
	destdir = 0;
    }

  if (result == 0)
    {
      fprintf (stderr, "Don't know where to install the font!\n");
      EXIT (1);
    }

  return (result);
}

/* _________________________________________________________________________

   MAKING FILE NAMES.
   _________________________________________________________________________ */


static char *
make_gf_filename (const char *	name,
		  int		dpi)

{
  char result[_MAX_PATH];
  sprintf (result, "%s.%dgf", name, dpi);
  return (strdup (result));
}

static char *
make_pk_filename (const char *	name,
		  int		bdpi,
		  int		dpi)

{
  char *cp_result;
  const char *cp_name_template;
  char result[_MAX_PATH];
  char name_template[_MAX_FNAME];

  cp_result = result;
  cp_name_template = get_cfg_value ("MakeTeXPK", "PK File Name",
				    name_template, sizeof (name_template),
				    "%f.%dpk");

  while (*cp_name_template)
    {
      if (*cp_name_template == '%')
	{
	  switch (cp_name_template[1])
	    {
	    default:
	      *cp_result = 0;
	      break;
	    case '%':
	      *cp_result++ = '%';
	      *cp_result = 0;
	      break;
	    case 'f':
	      sprintf (cp_result, "%s", name);
	      break;
	    case 'b':
	      sprintf (cp_result, "%d", bdpi);
	      break;
	    case 'd':
	      sprintf (cp_result, "%d", dpi);
	      break;
	    }
	  cp_result += strlen (cp_result);
	  cp_name_template += 2;
	}
      else
	*cp_result++ = *cp_name_template++;
    }
  *cp_result = 0;

  return (strdup (result));
}

static char *
make_dest_path (const char *	destdir,
		const char *	pkname)

{
  char result[_MAX_PATH];
  sprintf (result, "%s\\%s", destdir, pkname);
  return (strdup (result));
}

/* _________________________________________________________________________

   MAKING A MODE NAME.
   _________________________________________________________________________ */


static char *
make_mode_name (int	bdpi,
		char **	magnification)

{
  char *buf;
  const char *mode;
  switch (bdpi)
    {
    case 300: mode = "imagen"; break;
    case 600:
      buf = malloc (strlen (*magnification) + 4 + 1);
      mode = "imagen";
      sprintf (buf, "2*(%s)", *magnification);
      *magnification = buf;
      break;
    case 200: mode = "FAX"; break;
    case 360: mode = "nextII"; break;
    case 400: mode = "nexthi"; break;
    case 100: mode = "nextscreen"; break;
    case 72: mode = "seventwo"; break;
    case 635: mode = "linolo"; break;
    case 1270: mode = "linohi"; break;
    case 2540: mode = "linosuper"; break;
    default:
      fprintf (stderr, "I don't know the mode for %d\n", bdpi);
      fprintf (stderr, "Have your system admin update MakeTeXPK.c\n");
      EXIT (1);
    }
  return (strdup (mode));
}

/* _________________________________________________________________________

   INVOKING METAFONT AND GFTOPK.
   _________________________________________________________________________ */

static int
invoke_metafont (const char *	mode,
		 const char *	magnification,
		 const char *	name)

{
  char command_line[2 * _MAX_PATH + 100];
  char mode_def[50];
  char mag_def[50];
  int pargc;
  const char *pargv[20];
  int rc;

  if (! find_executable ("virmf.exe", command_line))
    {
      fprintf (stderr, "virmf.exe not found!\n");
      if (debug_p)
	fprintf (stderr, "Intermediate files were *not* removed!\n");
      else
	leave_temporary_directory ();
      EXIT (1);
    }

  sprintf (mode_def, "\\mode:=%s;", mode);
  sprintf (mag_def, "\\mag:=%s;", magnification);
  
  pargc = 0;
  pargv[pargc++] = command_line;
  pargv[pargc++] = mode_def;
  pargv[pargc++] = mag_def;
  pargv[pargc++] = "batchmode;";
  pargv[pargc++] = "input";
  pargv[pargc++] = name;
  pargv[pargc++] = 0;
  
  if (! quiet_p || print_only_p)
    printf ("%s %s %s batchmode; input %s\n",
	    command_line, mode_def, mag_def, name);

  if (print_only_p)
    rc = 0;
  else
    {
      _flushall ();
      rc = _spawnv (_P_WAIT, command_line, pargv);
    }

  if (rc < 0)
    {
      perror (command_line);
      leave_temporary_directory ();
      EXIT (1);
    }

  if (rc > 0)
    {
      fprintf (stderr, "METAFONT failed on %s! Return code: %d\n", name, rc);
      if (debug_p)
	fprintf (stderr, "Intermediate files were *not* removed!\n");
      else
	leave_temporary_directory ();
      EXIT (1);
    }

  return (rc);
}

static int
invoke_gftopk (const char *gfname,
	       const char *pkname)

{
  char command_line[3 * _MAX_PATH + 3];
  int pargc;
  const char *pargv[20];
  int rc;

  if (! print_only_p && _access (gfname, 0) < 0)
    {
      fprintf (stderr, "Cannot access GF file %s!\n", gfname);
      if (debug_p)
	fprintf (stderr, "Intermediate files were *not* removed!\n");
      else
	leave_temporary_directory ();
      EXIT (1);
    }

  if (! find_executable ("gftopk.exe", command_line))
    {
      fprintf (stderr, "gftopk.exe not found!\n");
      if (debug_p)
	fprintf (stderr, "Intermediate files were *not* removed!\n");
      else
	leave_temporary_directory ();
      EXIT (1);
    }

  pargc = 0;
  pargv[pargc++] = command_line;
  pargv[pargc++] = gfname;
  pargv[pargc++] = pkname;
  pargv[pargc++] = 0;
  
  if (verbose_p || print_only_p)
    printf ("%s %s %s\n", command_line, gfname, pkname);
  if (print_only_p)
    rc = 0;
  else
    {
      _flushall ();
      rc = _spawnv (_P_WAIT, command_line, pargv);
    }
  
  if (rc < 0)
    {
      perror (command_line);
      leave_temporary_directory ();
      EXIT (1);
    }

  if (rc > 0)
    {
      fprintf (stderr, "GFtoPK failed on %s! Return code: %d\n", gfname, rc);
      if (debug_p)
	fprintf (stderr, "Intermediate files were *not* removed!\n");
      else
	leave_temporary_directory ();
      EXIT (1);
    }
  
  return (rc);
}

/* _________________________________________________________________________

   INSTALLING PK FONTS.
   _________________________________________________________________________ */


static void
install_pk_font (const char *	pkname,
		 const char *	destdir,
		 const char *	destname)

{
  if (! print_only_p && _access (pkname, 0) < 0)
    {
      fprintf (stderr, "Cannot access PK file %s\n", pkname);
      if (debug_p)
	fprintf (stderr, "Intermediate files were *not* removed!\n");
      else
	leave_temporary_directory ();
      EXIT (1);
    }
  if (print_only_p || verbose_p)
    printf ("move %s %s\n", pkname, destname);
  if (! print_only_p)
    {
      if (make_path (destdir))
	{
	  if (rename (pkname, destname) != 0)
	    {
	      perror (pkname);
	      leave_temporary_directory ();
	      EXIT (1);
	    }
	}
    }
}

/* _________________________________________________________________________

   MAIN.
   _________________________________________________________________________ */


int
main (int	argc,
      char **	argv)

{
  char *gfname;
  char *pkname;
  char *destname;

  /* Get command line options. */
  c4pargv[0] = argv[0];
  get_options (argc, argv);

  /* Make a mode name if none was specified. */
  if (mode != 0 && strcmp (mode, "default") == 0)
    mode = 0;
  if (mode == 0)
    mode = make_mode_name (bdpi, &magnification);

  /* Make destination directory name if none was specified. */
  if (destdir == 0)
    destdir = create_dest_directory (name, bdpi, dpi, mode);

  /* Make GF file name and PK file name. */
  gfname = make_gf_filename (name, dpi);
  pkname = make_pk_filename (name, bdpi, dpi);

  /* Make full destination path name. */
  destname = make_dest_path (destdir, pkname);

  /* Quit, if destination font file already exists. */
  if (_access (destname, 0) == 0)
    {
      if (! quiet_p)
	printf ("%s already exists!\n", destname);
      EXIT (0);
    }

  if (verbose_p && ! print_only_p)
    printf ("Creating %s...\n", pkname);

  /* Create temporary directory; make it current. */
  enter_temporary_directory ("pk", print_only_p);

  /* Invoke METAFONT and GFtoPK to make a PK font. */
  invoke_metafont (mode, magnification, name);
  invoke_gftopk (gfname, pkname);

  /* Install PK font file in destination directory. */
  install_pk_font (pkname, destdir, destname);
  
  /* Remove temporary directory. */
  leave_temporary_directory ();

  EXIT (0);
  return (0);
}

/* MakeTeXPK.c ends here */
