/*-
******************************************************************************
******************************************************************************
**
**  ARCHIVE HEADER INFORMATION
**
**  @C-file{
**      FILENAME    = "specific.h",
**      VERSION     = "1.00",
**      DATE        = "",
**      TIME        = "",
**
**      AUTHOR      = "Niel Kempson",
**      ADDRESS     = "25 Whitethorn Drive, Cheltenham, GL52 5LL, England",
**      TELEPHONE   = "+44-242 579105",
**      EMAIL       = "kempson@tex.ac.uk (Internet)",
**
**      SUPPORTED   = "yes",
**      ARCHIVED    = "tex.ac.uk, ftp.tex.ac.uk",
**      KEYWORDS    = "VVcode",
**
**      CODETABLE   = "ISO/ASCII",
**      CHECKSUM    = "51492 1481 5732 57976",
**
**      DOCSTRING   = { This file is part of VVcode.
**                  }
**  }
**
**  MODULE CONTENTS
**
**      apply_defaults      -   Apply default file name and extension to a
**                              full file specification if either of these 
**                              components is missing.
**      confirm_yesno       -   Display a message to the user and wait for a
**                              yes/no answer.
**      examine_file        -   Examine a file and determine its key features, 
**                              including mode, format, record length and 
**                              timestamp.
**      explain_error       -   Explain the reason for an error.
**      f_open_in           -   Open a file for input using the appropriate 
**                              mode, format etc for the file.
**      f_open_out          -   Open a file for output using the appropriate
**                              mode, format etc for the file.
**      file_exists         -   Determine whether a file exists.
**      force_file_ext      -   Force the file extension of a full file
**                              specification to a specified value.
**      is_a_file           -   Determine whether a file stream is connected
**                              to a real file rather than a character
**                              device, pipe etc.
**      legal_filespec      -   Takes an arbitrary string which may a file 
**                              specification from another operating system
**                              and manipulates it to produce a legal file 
**                              specification for the current operating
**                              system.
**      make_filespec       -   Construct a full file specification from
**                              component parts.
**      prompt_for_string   -   Present a prompt string and accept a string 
**                              from the user.
**      read_bytes          -   Read bytes from a currently open file.
**      read_line           -   Read a line from a currently open (text) file.
**      read_record         -   Read a record from a currently open file.
**      scan_cmd_line       -   [tbs]
**      set_ftime           -   Set the timestamp of a file to a specified 
**                              value.
**      set_pipe_mode       -   Set the mode of a file stream connected to a
**                              pipe, character device, redirected
**                              stdin/stdout/stderr, in fact any non-file.
**      split_file_spec     -   Split a full file specification into its
**                              component parts.
**      tz_offset           -   Determine the offset of local time from 
**                              Greenwich Mean Time (Coordinated Universal
**                              Time) at a specified date and time.  
**      user_message        -   Present a message to the user.
**      vv_exit             -   Exit the program, returning the appropriate
**                              status to the operating system.
**      write_bytes         -   Write bytes to a currently open file.
**      write_record        -   Write a record to a currently open file.
**
**  COPYRIGHT
**
**      Copyright (c) 1991-1993 by Niel Kempson <kempson@tex.ac.uk>
**
**      This program is free software; you can redistribute it and/or
**      modify it under the terms of the GNU General Public License as
**      published by the Free Software Foundation; either version 1, or
**      (at your option) any later version.
**
**      This program is distributed in the hope that it will be useful,
**      but WITHOUT ANY WARRANTY; without even the implied warranty of
**      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
**      General Public License for more details.
**
**      You should have received a copy of the GNU General Public License
**      along with this program; if not, write to the Free Software
**      Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
**
**      In other words, you are welcome to use, share and improve this
**      program.  You are forbidden to forbid anyone else to use, share
**      and improve what you give them.   Help stamp out software-hoarding!  
**
**  CHANGE LOG
**
******************************************************************************
******************************************************************************
*/


/*-
**============================================================================
**
** FUNCTION
**
**      apply_defaults
**
** DESCRIPTION
**
**      Apply default file name and extension to a full file specification if
**      either of these components is missing.
**
** INPUT PARAMETERS
** 
**      full_spec       -   a string buffer of size (MAX_PATH + 1) containing
**                          the full file specification.
**      default_name    -   the default name component to be used if the full
**                          file specification does not contain a name
**                          component.  If there is no default name component,
**                          set this parameter to the null character pointer 
**                          defined by NULL.
**      default_ext     -   the default extension component to be used if the 
**                          full file specification does not contain an
**                          extension component.  If there is no default
**                          extension component, set this parameter to the 
**                          null character pointer defined by NULL.
**                          
** OUTPUT PARAMETERS
**
**      full_spec       -   a string buffer of size (MAX_PATH + 1), which on
**                          return will contain the full file specification
**                          with default name and extension components applied
**                          if appropriate.
**
** RETURN VALUE
**
**      None
**
**============================================================================
*/
void apply_defaults             ARGS ((CONST char *default_name,
                                    CONST char *default_ext,
                                    char *full_spec));



/*-
**============================================================================
**
** FUNCTION
**
**      confirm_yesno
**
** DESCRIPTION
**
**      Display a message to the user and wait for a yes/no answer.
**
** INPUT PARAMETERS
**
**      confirm_str     -   a string containing the message to be displayed.
**
** OUTPUT PARAMETERS
** 
**      None
**
** RETURN VALUE
**
**      TRUE if 'y' or 'Y' was entered, FALSE if 'n' or 'N' was entered.
**
**============================================================================
*/
Boolean confirm_yesno           ARGS ((CONST char *confirm_str));



/*-
**============================================================================
**
** FUNCTION
**
**      examine_file
**
** DESCRIPTION
**
**      Examine a file and determine its key features, including mode, format,
**      record length and timestamp.  The file information is passed via a 
**      File_Info structure.  The following fields of the structure are used:
**
** INPUT PARAMETERS
**
**      ex_file         -   the File_Info structure containing the feature
**                          information of the file to be examined.  The
**                          following structure fields are used for input:
**
**        .file_spec    -   the file specification of the file to be examined
**
** OUTPUT PARAMETERS
** 
**      ex_file         -   the File_Info structure whose elements will be
**                          updated to reflect the file features.  The
**                          following structure fields are used for output:
**
**        .mode         -   the mode of the file to be examined (binary/text).
**                          If the mode cannot be determined it shall be set
**                          to INV_MODE.
**        .format       -   the format of the file to be examined (fixed,
**                          stream or variable).  If the format cannot be 
**                          determined or is not supported, it shall be set
**                          to INV_FMT.
**        .max_rec_len  -   the maximum allowable record length.  If the
**                          maximum allowable record length cannot be
**                          determined or is not supported, it shall be set
**                          to INV_RECORD_LEN.
**        .lng_rec_len  -   the length of the longest record in the file.  If 
**                          the longest record length cannot be determined or
**                          is not supported, it shall be set to 
**                          INV_RECORD_LEN.
**        .mod_time    -    the modification time of the file in Unix format.
**                          If the time cannot be determined or is not
**                          supported, it shall be set to INV_TIMESTAMP.
**
** RETURN VALUE
**
**      None
**
**============================================================================
*/
void examine_file               ARGS ((File_Info *ex_info));



/*-
**============================================================================
**
** FUNCTION
**
**      explain_error
**
** DESCRIPTION
**
**      Explain the reason for a detected error.
**
** INPUT PARAMETERS
**
**      None
**
** OUTPUT PARAMETERS
** 
**      None
**
** RETURN VALUE
**
**      None
**
**============================================================================
*/
void explain_error              ARGS ((void));



/*-
**============================================================================
**
** FUNCTION
**
**      f_open_in
**
** DESCRIPTION
**
**      Open a file for input using the appropriate mode, format etc for the
**      file.
**
** INPUT PARAMETERS
**
**      ip_file       -     the File_Info structure whose elements will be
**                          used to open the file.  The following structure 
**                          fields are used for input:
**
**        .mode         -   the mode of the file (binary/text).
**        .format       -   the format of the file (fixed, stream or variable).
**        .max_rec_len  -   the maximum allowable record length.  If this
**                          field is set to INV_RECORD_LEN, a sensible default
**                          shall be chosen if needed and possible.
**        .lng_rec_len  -   the length of the longest record in the file.  If
**                          this field is set to INV_RECORD_LEN, a sensible
**                          default shall be chosen if needed and possible.
**
** OUTPUT PARAMETERS
** 
**      None
**
** RETURN VALUE
**
**      If the file was opened successfully, a (FILE *) pointer is returned,
**      otherwise (FILE *) NULL in the event of failure.
**
**============================================================================
*/
FILE *f_open_in                 ARGS ((File_Info *ip_file));



/*-
**============================================================================
**
** FUNCTION
**
**      f_open_out
**
** DESCRIPTION
**
**      Open a file for output using the appropriate mode, format etc for the
**      file.
**
** INPUT PARAMETERS
**
**      op_file       -     the File_Info structure whose elements will be
**                          used to open the file.  The following structure 
**                          fields are used for input:
**
**        .mode         -   the mode of the file (binary/text).
**        .format       -   the format of the file (fixed, stream or variable).
**        .max_rec_len  -   the maximum allowable record length.  If this
**                          field is set to INV_RECORD_LEN, a sensible default
**                          shall be chosen if needed and possible.
**        .lng_rec_len  -   the length of the longest record in the file.  If
**                          this field is set to INV_RECORD_LEN, a sensible
**                          default shall be chosen if needed and possible.
**
**      overwrite_files -   flag to control the overwriting of existing files.
**                          If TRUE, existing files will be overwritten,
**                          otherwise the user will be asked to confirm that
**                          the files should be overwritten.
** 
** OUTPUT PARAMETERS
** 
**      None
**
** RETURN VALUE
**
**      If the file was opened successfully, a (FILE *) pointer is returned,
**      otherwise (FILE *) NULL in the event of failure.
**
**============================================================================
*/
FILE *f_open_out                ARGS ((CONST Int16 overwrite_files,
                                    File_Info *op_file));



/*-
**============================================================================
**
** FUNCTION
**
**      file_exists
**
** DESCRIPTION
**
**      Determine whether a file exists.  Access privileges (e.g. R, RW) are
**      not checked.  
**
** INPUT PARAMETERS
** 
**      file_spec       -   the file specification of the file to be checked.
**
** OUTPUT PARAMETERS
**
**      None
**
** RETURN VALUE
**
**      TRUE if the file exists, FALSE if not.
**
**============================================================================
*/
Boolean file_exists             ARGS ((CONST char *file_spec));



/*-
**============================================================================
**
** FUNCTION
**
**      force_file_ext
**
** DESCRIPTION
**
**      Force the file extension of a full file specification to a specified
**      value.
**
** INPUT PARAMETERS
** 
**      full_spec       -   a string buffer of size (MAX_PATH + 1) containing
**                          the full file specification.
**      forced_ext      -   the new extension component to be used in the 
**                          full file specification.  If there is no new
**                          extension component, set this parameter to the 
**                          null character pointer defined by NULL.
**                          
** OUTPUT PARAMETERS
**
**      full_spec       -   a string buffer of size (MAX_PATH + 1), which on
**                          return will contain the full file specification
**                          with the new extension component applied.
**
** RETURN VALUE
**
**      None
**
**============================================================================
*/
void force_file_ext             ARGS ((CONST char *forced_ext,
                                    char *full_spec));



/*-
**============================================================================
**
** FUNCTION
**
**      is_a_file
**
** DESCRIPTION
**
**      Determine whether an already open stream is connected to a real file
**      rather than a character mode device or pipe.
**
** INPUT PARAMETERS
** 
**      ex_file
**
** OUTPUT PARAMETERS
**
**      None
**
** RETURN VALUE
**
**      TRUE if the stream is connected to a real file, FALSE if not.
**
**============================================================================
*/
Boolean is_a_file               ARGS ((CONST File_Info *ex_file));



/*-
**============================================================================
**
** FUNCTION
**
**      legal_filespec
**
** DESCRIPTION
**
**      Takes an arbitrary string which may a file specification from another
**      operating system and manipulates it to produce a legal file
**      specification for the current operating system.  The resultant file
**      specification must not include any preamble or postamble components
**      (i.e. name and/or extension only).
**
**      The string may be truncated, characters translated, or even untouched.
**
** INPUT PARAMETERS
** 
**      hdrf_spec       -   the arbitrary string which is usually a file
**                          specification from another operating system.
**
** OUTPUT PARAMETERS
**
**      None
**
** RETURN VALUE
**
**      A pointer to the legal file specification string.  The string is held
**      in a static buffer and will be overwritten by successive calls to this
**      function.
**
**============================================================================
*/
char *legal_filespec            ARGS ((CONST char *hdrf_spec));



/*-
**============================================================================
**
** FUNCTION
**
**      make_file_spec
**
** DESCRIPTION
**
**      Construct a full file specification from component parts.
**
** INPUT PARAMETERS
** 
**      preamble        -   the preamble component of the file specification.
**                          It shall not be necessary for the string to end 
**                          in one of the characters '\' or ':'.  If
**                          there is no name component, set this parameter to
**                          the null character pointer defined by NULL.
**      name            -   the name component of the file specification.  If
**                          there is no name component, set this parameter to
**                          the null character pointer defined by NULL.
**      extension       -   the extension component of the file
**                          specification.  It shall not be necessary for the
**                          string to begin with a '.'.  If there is no
**                          extension component, set this parameter to the
**                          null character pointer defined by NULL.
**      postamble       -   the postamble component of the file
**                          specification.  MS-DOS doesn't support the
**                          postamble component of the VVcode file
**                          specification model so this parameter is ignored.
**                          
** OUTPUT PARAMETERS
**
**      full_spec       -   a string buffer of size (MAX_PATH + 1), which on
**                          return will contain the full file specification.
**
** RETURN VALUE
**
**      None
**
**============================================================================
*/
void make_file_spec             ARGS ((CONST char *preamble,
                                    CONST char *name,
                                    CONST char *extension,
                                    CONST char *postamble,
                                    char *full_spec));



/*-
**============================================================================
**
** FUNCTION
**
**      prompt_for_string
**
** DESCRIPTION
**
**      Display a message to the user and accept a string.
**
** INPUT PARAMETERS
**
**      prompt_str      -   a string containing the message to be displayed.
**      buf_size        -   the size of the buffer which will contain the
**                          returned string.
**
** OUTPUT PARAMETERS
** 
**      return_buffer   -   a buffer to contain the string entered by the
**                          user.
**
** RETURN VALUE
**
**      None
**
**============================================================================
*/
void prompt_for_string          ARGS ((CONST char *confirm_str,
                                    CONST Int16 buf_size,
                                    char *return_buffer));



/*-
**============================================================================
**
** FUNCTION
**
**      read_bytes
**
** DESCRIPTION
**
**      Read bytes from a currently open file.  This function is called to 
**      read stream and fixed length record format files.  If the operating 
**      system does not support stream or fixed length record format files,
**      this function will never be called, BUT IT MUST BE PRESENT to avoid
**      linker errors.
**
** INPUT PARAMETERS
** 
**      ip_file       -     the File_Info structure whose elements will be
**                          used to read the record.  The following structure
**                          fields are used for input:
**
**        .file_ptr     -   the FILE structure of the stream
**
**      max_bytes       -   the maximum number of bytes that may be read
**
** OUTPUT PARAMETERS
**
**      buffer          -   a buffer of size 'max_bytes' or longer which on
**                          return will contain the bytes read from the file
**
** RETURN VALUE
**
**       #              -   the number of bytes returned in the buffer
**                          (excluding the trailing '\0').
**      -1              -   on EOF
**      -2              -   if an error was detected
**
**============================================================================
*/
Int32 read_bytes                ARGS ((CONST Int32 max_bytes,
                                    char *buffer,
                                    File_Info *ip_file));



/*-
**============================================================================
**
** FUNCTION
**
**      read_line
**
** DESCRIPTION
**
**      Read a line from a currently open file.  This function is called to 
**      read text files.
**
** INPUT PARAMETERS
** 
**      ip_file       -     the File_Info structure whose elements will be
**                          used to read the record.  The following structure
**                          fields are used for input:
**
**        .file_ptr     -   the FILE structure of the stream
**
**      max_bytes       -   the maximum number of bytes that may be read
**
** OUTPUT PARAMETERS
**
**      buffer          -   a buffer of size 'max_bytes' or longer which on
**                          return will contain the line read from the file
**
** RETURN VALUE
**
**       #              -   the number of bytes returned in the buffer
**                          (excluding the trailing '\0').
**      -1              -   on EOF
**      -2              -   if an error was detected
**
**============================================================================
*/
Int32 read_line                 ARGS ((CONST Int32 max_bytes,
                                    char *buffer,
                                    File_Info *ip_file));



/*-
**============================================================================
**
** FUNCTION
**
**      read_record
**
** DESCRIPTION
**
**      Read a record from a currently open file.  This function is only
**      called to read variable length format files.  If the operating system
**      does not support variable length format files, this function will
**      never be called, BUT IT MUST BE PRESENT to avoid linker errors.
**
** INPUT PARAMETERS
** 
**      ip_file       -     the File_Info structure whose elements will be
**                          used to read the record.  The following structure
**                          fields are used for input:
**
**        .file_ptr     -   the FILE structure of the stream
**
**      max_bytes       -   the maximum number of bytes that may be read
**
** OUTPUT PARAMETERS
**
**      buffer          -   a buffer of size 'max_bytes' or longer which on
**                          return will contain the record read from the file
**
** RETURN VALUE
**
**       #              -   the number of bytes returned in the buffer
**                          (including an appended '\n' characters but
**                          excluding the trailing '\0').
**      -1              -   on EOF
**      -2              -   if an error was detected
**
**============================================================================
*/
Int32 read_record               ARGS ((CONST Int32 max_bytes,
                                    char *buffer,
                                    File_Info *ip_file));



/*-
**============================================================================
**
** FUNCTION
**
**      scan_cmd_line
**
** DESCRIPTION
**
**      [tbs]
**
** INPUT PARAMETERS
** 
**      qargc           -   the number of command line arguments to be parsed
**                          (a copy of the main() argc parameter).
**      qargv           -   an array of strings containing the command line
**                          arguments to be parsed (a copy of the main() argv
**                          parameter).
**      q_intro_str     -   a string containing the valid single characters
**                          which may be used to introduce a command
**                          qualifier (e.g. "/-" for MS-DOS, "-" for Unix).
**      q_sep_str       -   a string containing the valid single characters
**                          which may be used to separate a command qualifier
**                          from its value (e.g. "=:-" for MS-DOS,
**                          " " for Unix).
**
** OUTPUT PARAMETERS
**
**      qual_array      -   the array of Qualifier_Struct structures which
**                          contain the status information for each of the
**                          valid command qualifiers
**      
** RETURN VALUE
**
**      None
**
**============================================================================
*/
void scan_cmd_line              ARGS ((CONST int qargc,
                                    CONST char *qargv[],
                                    CONST char *q_intro_str,
                                    CONST char *q_sep_str,
                                    Qualifier_Struct *qual_array,
                                    Qualifier_Struct *qual_ipfile,
                                    Qualifier_Struct *qual_opfile));



/*-
**============================================================================
**
** FUNCTION
**
**      set_ftime
**
** DESCRIPTION
**
**      Set the timestamp of a file to a specified value.  Operating systems
**      support different types of time, such as access, creation, revision or
**      modification time.  It is suggested that the creation and modification
**      time be set to this value if possible.
**
**      If the operating system does not support the setting of file
**      timestamps, this should be an empty function, BUT IT MUST BE PRESENT.
**
** INPUT PARAMETERS
** 
**      target_file   -     the File_Info structure containing the feature
**                          information of the file whose timestamp is to be
**                          set.  The following structure fields are used
**                          for input:
**
**        .file_spec    -   the file specification of the file.
**        .mod_time     -   the modification time of the file in Unix format.
**
** OUTPUT PARAMETERS
**
**      None
**
** RETURN VALUE
**
**      None
**
**============================================================================
*/
void set_ftime                  ARGS ((File_Info *target_file));



/*-
**============================================================================
**
** FUNCTION
**
**      set_pipe_mode
**
** DESCRIPTION
**
**      Set the mode of an already open file stream which is a
**      character mode device or pipe rather than a real file.
**
**      If text/binary mode has no meaning or mode setting is not supported,
**      this function may be empty, BUT IT MUST BE PRESENT.
**
** INPUT PARAMETERS
** 
**      target_file   -     the File_Info structure containing the feature
**                          information of the file whose mode is to be
**                          changed.  The following structure fields are used
**                          for input:
**
**        .file_spec    -   the file specification of the open file.
**        .file_ptr     -   the (FILE *) pointer to the open file
**        .mode         -   the mode of the file to be examined (binary/text).
**
** OUTPUT PARAMETERS
**
**      None
**
** RETURN VALUE
**
**      None
**
**============================================================================
*/
void set_pipe_mode              ARGS ((File_Info *target_file));



/*-
**============================================================================
**
** FUNCTION
**
**      split_file_spec
**
** DESCRIPTION
**
**      Split a full file specification into its component parts and return
**      a bit field indicating which components of the file specification
**      were present.
**
** INPUT PARAMETERS
**
**      full_spec       -   a string buffer of size (MAX_PATH + 1), which
**                          contains the full file specification.
**
** OUTPUT PARAMETERS
** 
**      preamble        -   a buffer of size (MAX_PREAMBLE + 1) to hold the
**                          preamble component of the file specification.
**                          The string shall end in one of the characters
**                          '\' or ':'.  If there is no preamble component, 
**                          this parameter shall be set to the null character
**                          pointer defined by NULL.
**      name            -   a buffer of size (MAX_NAME + 1) to hold the name
**                          component of the file specification.  If there is
**                          no name component, this parameter shall be set to
**                          the null character pointer defined by NULL.
**      extension       -   a buffer of size (MAX_EXT + 1) to hold the
**                          extension component of the file specification.
**                          The string shall begin with '.'  If there
**                          is no extension component, this parameter shall
**                          be set to the null character pointer defined by
**                          NULL.
**      postamble       -   a buffer of size (MAX_POSTAMBLE + 1) to hold the
**                          postamble component of the file specification.
**                          MS-DOS doesn't support the postamble component of
**                          the VVcode file specification model so this 
**                          parameter is ignored.
**
** RETURN VALUE
**
**      An Int16 value indicating which components were found in the file
**      specification.  The value is composed of the following macros:
**
**          FS_NOTHING      -   no components were present (i.e. an empty file
**                              or null specification)
**          FS_PREAMBLE     -   a preamble component was present
**          FS_NAME         -   a name component was present
**          FS_EXTENSION    -   an extension component was present
**          FS_POSTAMBLE    -   a postamble component was present
**
**      If the full file specification contains more than one component, these
**      values are OR'd together.
**
**============================================================================
*/
Unsigned16 split_file_spec      ARGS ((CONST char *full_spec,
                                    char *preamble,
                                    char *name,
                                    char *extension,
                                    char *postamble));



/*-
**============================================================================
**
** FUNCTION
**
**      tz_offset
**
** DESCRIPTION
**
**      Determine the offset of local time from Greenwich Mean Time
**      (Coordinated Universal Time) at a specified date and time.  The date
**      and time is specified because the offset may vary due to different
**      implementations of summer (daylight saving) time around the world.
**
**      If the operating system does not support the concept of time zones,
**      this function should return a zero offset.
**
** INPUT PARAMETERS
** 
**      the_time        -   the GMT time in Unix time_t format at which the
**                          offset is to be computed.
**
** OUTPUT PARAMETERS
**
**      None
**
** RETURN VALUE
**
**      The offset of local time from GMT in seconds.  Positive values denote
**      times WEST of (i.e. behind) GMT, and negative values are EAST (i.e.
**      ahead) of GMT.
**
**============================================================================
*/
TIME_T tz_offset                ARGS ((TIME_T the_time));



/*-
**============================================================================
**
** FUNCTION
**
**      user_message
**
** DESCRIPTION
**
**      [tbs]
**
** INPUT PARAMETERS
**
**      [tbs]
**
** OUTPUT PARAMETERS
**
**      [tbs]
**
** RETURN VALUE
**
**      [tbs]
**
**============================================================================
*/
void user_message               ARGS ((CONST Int16 status,
				    CONST char *msg_str,
				    File_Info *log_file));



/*-
**============================================================================
**
** FUNCTION
**
**      vv_exit
**
** DESCRIPTION
**
**      Exit the program, returning the appropriate status to the operating
**      system.
**
** INPUT PARAMETERS
** 
**      status          -   the VVcode status value which will be used to
**                          determine the exit status passed to the operating
**                          system
**
** OUTPUT PARAMETERS
**
**      None
**
** RETURN VALUE
**
**      None
**
**============================================================================
*/
void vv_exit                    ARGS ((void));


  
/*-
**============================================================================
**
** FUNCTION
**
**      write_bytes
**
** DESCRIPTION
**
**      Write bytes to a currently open file.
**
** INPUT PARAMETERS
** 
**      op_file       -     the File_Info structure whose elements will be
**                          used to write the record.  The following structure
**                          fields are used for input:
**
**        .file_ptr     -   the FILE structure of the stream
**        .max_rec_len  -   the maximum allowable record length.
**
**      buffer          -   a buffer of size 'rec_len' or longer which
**                          contains the record to be written to the file
**      rec_len         -   the number of bytes in the record
**
** OUTPUT PARAMETERS
**
**      None
**
** RETURN VALUE
**
**      None
**
**============================================================================
*/
void write_bytes                ARGS ((CONST Int32 rec_len,
                                    CONST char *buffer,
                                    File_Info *op_file));



/*-
**============================================================================
**
** FUNCTION
**
**      write_record
**
** DESCRIPTION
**
**      Write a record to a currently open file.  This function is only
**      called to write variable length format files.  If the operating
**      system does not support variable length format files, this function
**      will never be called, BUT IT MUST BE PRESENT to avoid linker errors.
**
** INPUT PARAMETERS
** 
**      op_file       -     the File_Info structure whose elements will be
**                          used to write the record.  The following structure
**                          fields are used for input:
**
**        .file_ptr     -   the FILE structure of the stream
**        .max_rec_len  -   the maximum allowable record length.
**
**      buffer          -   a buffer of size 'rec_len' or longer which
**                          contains the record to be written to the file
**      rec_len         -   the number of bytes in the record
**
** OUTPUT PARAMETERS
**
**      None
**
** RETURN VALUE
**
**      None
**
**============================================================================
*/
void write_record               ARGS ((CONST Int32 rec_len,
                                    CONST char *buffer,
                                    File_Info *op_file));



